/*
 * Decompiled with CFR 0.152.
 */
package by.ely.authlib;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Iterables;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import com.mojang.authlib.GameProfile;
import com.mojang.authlib.minecraft.MinecraftProfileTexture;
import com.mojang.authlib.properties.Property;
import com.mojang.authlib.yggdrasil.response.MinecraftTexturesPayload;
import com.mojang.util.UUIDTypeAdapter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Type;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class ElyTextureService {
    private static final String SKIN_SYSTEM_BASE_URL = "http://skinsystem.ely.by";
    private static final String OFFLINE_SKIN_REQUEST_URL = "http://skinsystem.ely.by/skins/%s?auth_lib=%s";
    private static final String ONLINE_SKIN_REQUEST_URL = "http://skinsystem.ely.by/textures/%s?auth_lib=%s";
    private static final Logger LOGGER = LogManager.getLogger();
    private final Gson gson = new GsonBuilder().registerTypeAdapter(UUID.class, (Object)new UUIDTypeAdapter()).create();
    private final Cache<String, Map<MinecraftProfileTexture.Type, MinecraftProfileTexture>> onlineTextureCache = CacheBuilder.newBuilder().expireAfterWrite(30L, TimeUnit.MINUTES).softValues().build();
    static final boolean debug = Boolean.parseBoolean(System.getProperty("ely.authlib.debug", "false"));
    private static final List<String> FOREGROUND_THREADS = Arrays.asList("Client thread", "Render thread", "Server thread");
    private static final String LIB_VERSION;

    public Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> getTextures(GameProfile profile) {
        Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> textureMap;
        ElyTextureService.debug("Getting textures for {}", ElyTextureService.ofProfile(profile));
        long start = System.nanoTime();
        try {
            textureMap = this.getTextureMap(profile);
        }
        catch (RuntimeException rE) {
            ElyTextureService.error("Could not get texture map of " + ElyTextureService.ofProfile(profile), rE);
            throw rE;
        }
        ElyTextureService.debug("Done for {} in {} ms. Textures: {}", ElyTextureService.ofProfile(profile), (double)(System.nanoTime() - start) / 1000000.0, textureMap);
        return ElyTextureService.ofTextureMap(textureMap);
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> getTextureMap(GameProfile profile) {
        MinecraftTexturesPayload payload;
        String profileName = profile.getName();
        Property textureProperty = (Property)Iterables.getFirst((Iterable)profile.getProperties().get("textures"), null);
        if (textureProperty == null) {
            ElyTextureService.debug("No textures in the profile of {}", profileName);
            return this.getElyTexture(profileName, null);
        }
        try {
            payload = this.decodePayload(textureProperty);
        }
        catch (RuntimeException e) {
            if (debug) {
                ElyTextureService.debug("Invalid texture payload for {}: {}", profileName, textureProperty);
            }
            ElyTextureService.error("Could not decode textures payload of " + profileName, e);
            payload = null;
        }
        if (profile.getProperties().containsKey("ely")) {
            ElyTextureService.debug("Ely flag found in the profile of {}", ElyTextureService.ofProfile(profile));
            return ElyTextureService.ofPayload(payload);
        }
        if (profile.isComplete()) {
            ElyTextureService.debug("Received complete profile of {} with textures: {}", profileName, payload);
            return this.getTexturesUsingPayload(profile, payload);
        }
        if (payload == null) {
            ElyTextureService.debug("Payload is invalid and profile is incomplete; returning as-is: {}", profile);
        } else {
            ElyTextureService.debug("Got incomplete profile; returning it as-is: {}", profile);
        }
        return ElyTextureService.ofPayload(payload);
    }

    public MinecraftTexturesPayload decodePayload(Property textureProperty) {
        String decode = new String(Base64.getDecoder().decode(textureProperty.getValue()), StandardCharsets.UTF_8);
        MinecraftTexturesPayload payload = (MinecraftTexturesPayload)this.gson.fromJson(decode, MinecraftTexturesPayload.class);
        return (MinecraftTexturesPayload)Validate.notNull((Object)payload, (String)"payload", (Object[])new Object[0]);
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> getTexturesUsingPayload(GameProfile profile, MinecraftTexturesPayload payload) {
        if (ElyTextureService.isNullSkin(payload) || ElyTextureService.isOwnSkin(profile, payload)) {
            ElyTextureService.debug("Will use Ely for {}", profile.getName());
            return this.getElyTexture(profile.getName(), payload);
        }
        ElyTextureService.debug("Texture payload of {} refers to {}; won't use Ely", profile.getName(), payload.getProfileName() == null ? "(no profile name)" : payload.getProfileName());
        return payload.getTextures();
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> getElyTexture(String profileName, MinecraftTexturesPayload payload) {
        if (profileName == null) {
            ElyTextureService.debug("Cannot proceed with null profile name", new Object[0]);
            return Collections.emptyMap();
        }
        if (ElyTextureService.isForegroundThread()) {
            return this.getElyTextureOffline(profileName, payload);
        }
        Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> elyTextures = this.getElyTextureOnline(profileName);
        if (this.detectIfNotFoundOnline(elyTextures)) {
            ElyTextureService.debug("Ely does not contain skins for {}", profileName);
            return ElyTextureService.ofPayload(payload);
        }
        return elyTextures;
    }

    private boolean detectIfNotFoundOnline(Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> onlineResult) {
        if (onlineResult == null || onlineResult.isEmpty() || !onlineResult.containsKey((Object)MinecraftProfileTexture.Type.SKIN)) {
            return true;
        }
        MinecraftProfileTexture skinTexture = onlineResult.get((Object)MinecraftProfileTexture.Type.SKIN);
        return skinTexture.getUrl().startsWith("http://skins.minecraft.net");
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> getElyTextureOffline(String profileName, MinecraftTexturesPayload payload) {
        ElyTextureService.debug("Requesting textures offline for {}", profileName);
        String hash = payload != null && payload.getTextures().get((Object)MinecraftProfileTexture.Type.SKIN) != null ? payload.getTextures().get((Object)MinecraftProfileTexture.Type.SKIN).getHash() : profileName;
        return ElyTextureService.ofTexture(String.format(OFFLINE_SKIN_REQUEST_URL, profileName, LIB_VERSION), hash);
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> getElyTextureOnline(String profileName) {
        ElyTextureService.debug("Requesting textures online for {}", profileName);
        Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> textureMap = (Map<MinecraftProfileTexture.Type, MinecraftProfileTexture>)this.onlineTextureCache.getIfPresent((Object)profileName);
        if (textureMap != null) {
            ElyTextureService.debug("Got online textures of {} from cache", profileName);
            return textureMap;
        }
        try {
            textureMap = this.fetchOnlineTextures(profileName);
        }
        catch (IOException ioE) {
            ElyTextureService.error(ioE);
        }
        if (textureMap != null) {
            this.onlineTextureCache.put((Object)profileName, textureMap);
        }
        return textureMap;
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> fetchOnlineTextures(String profileName) throws IOException {
        URL url = new URL(String.format(ONLINE_SKIN_REQUEST_URL, URLEncoder.encode(profileName, "UTF-8"), LIB_VERSION));
        try {
            if (debug) {
                return this.fetchDebug(url);
            }
            return this.fetchDefault(url);
        }
        catch (IOException ioE) {
            throw new IOException("could not fetch Ely textures of " + profileName + "; url: " + url, ioE);
        }
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> fetchDebug(URL url) throws IOException {
        String response = IOUtils.toString((URL)url, (Charset)StandardCharsets.UTF_8);
        ElyTextureService.debug("Request: {}, response: {}", url, response);
        try {
            return (Map)this.gson.fromJson(response, ElyTextureService.textureMapType());
        }
        catch (RuntimeException rE) {
            throw new IOException("could not parse response: \"" + response + "\"", rE);
        }
    }

    private Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> fetchDefault(URL url) throws IOException {
        Map map;
        block5: {
            InputStreamReader reader = null;
            try {
                reader = new InputStreamReader(url.openStream(), StandardCharsets.UTF_8);
                map = (Map)this.gson.fromJson((Reader)reader, ElyTextureService.textureMapType());
                if (reader == null) break block5;
            }
            catch (RuntimeException rE) {
                try {
                    throw new IOException("could not parse response", rE);
                }
                catch (Throwable throwable) {
                    if (reader != null) {
                        IOUtils.closeQuietly(reader);
                    }
                    throw throwable;
                }
            }
            IOUtils.closeQuietly((Reader)reader);
        }
        return map;
    }

    private static void debug(String s, Object ... o) {
        if (debug) {
            LOGGER.info(s, o);
        }
    }

    private static void error(String s, Throwable t) {
        LOGGER.error(s, t);
    }

    private static void error(Throwable t) {
        LOGGER.error((Object)t);
    }

    private static boolean isOwnSkin(GameProfile profile, MinecraftTexturesPayload payload) {
        return StringUtils.equalsIgnoreCase((CharSequence)profile.getName(), (CharSequence)(payload == null ? null : payload.getProfileName()));
    }

    private static boolean isNullSkin(MinecraftTexturesPayload payload) {
        return payload == null || payload.getProfileName() == null;
    }

    private static boolean isForegroundThread() {
        return FOREGROUND_THREADS.contains(Thread.currentThread().getName());
    }

    private static Object ofProfile(GameProfile profile) {
        if (profile == null) {
            return "(null profile)";
        }
        if (profile.getName() == null) {
            return profile;
        }
        return profile.getName();
    }

    public static Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> ofPayload(MinecraftTexturesPayload payload) {
        if (payload == null || payload.getTextures() == null) {
            return Collections.emptyMap();
        }
        return payload.getTextures();
    }

    private static Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> ofTextureMap(Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> textureMap) {
        if (textureMap == null) {
            return Collections.emptyMap();
        }
        return textureMap;
    }

    private static Map<MinecraftProfileTexture.Type, MinecraftProfileTexture> ofTexture(String url, String hash) {
        return Collections.singletonMap(MinecraftProfileTexture.Type.SKIN, new MinecraftProfileTexture(url, hash));
    }

    private static Type textureMapType() {
        return new TypeToken<Map<MinecraftProfileTexture.Type, MinecraftProfileTexture>>(){}.getType();
    }

    static {
        String version = ElyTextureService.class.getPackage().getImplementationVersion();
        LIB_VERSION = StringUtils.isBlank((CharSequence)version) ? "unknown" : version;
    }
}

